# WPS Office 2023/2019 专业增强版安装工具 - 最终完整修复版
# 彻底解决空值错误和路径查找问题
# 编码格式: UTF-8 with BOM

# 要求管理员权限
if (-not ([Security.Principal.WindowsPrincipal][Security.Principal.WindowsIdentity]::GetCurrent()).IsInRole([Security.Principal.WindowsBuiltInRole]::Administrator)) {
    Start-Process powershell.exe "-NoProfile -ExecutionPolicy Bypass -File `"$PSCommandPath`"" -Verb RunAs
    exit
}

# 隐藏控制台窗口
if (-not ("Console.Window" -as [type])) {
Add-Type -Name Window -Namespace Console -MemberDefinition '
[DllImport("Kernel32.dll")]
public static extern IntPtr GetConsoleWindow();
[DllImport("user32.dll")]
public static extern bool ShowWindow(IntPtr hWnd, Int32 nCmdShow);
'
}
$consolePtr = [Console.Window]::GetConsoleWindow()
[Console.Window]::ShowWindow($consolePtr, 0) | Out-Null

Add-Type -AssemblyName System.Windows.Forms
Add-Type -AssemblyName System.Drawing

[System.Windows.Forms.Application]::EnableVisualStyles()

# =============================================
# 全局变量
# =============================================
$global:isInstalling = $false
$global:form = $null
$global:installButton = $null
$global:install2019Button = $null
$global:logTextBox = $null
$global:progressBar = $null
$global:progressLabel = $null
$global:lastProgressText = ""
$global:selectedVersion = "2023"  # 默认安装版本

# =============================================
# 颜色定义
# =============================================
$primaryColor = [System.Drawing.Color]::FromArgb(22, 115, 255)      # 主色调
$primaryHover = [System.Drawing.Color]::FromArgb(0, 95, 230)        # 主色调悬停
$primaryPressed = [System.Drawing.Color]::FromArgb(0, 86, 179)      # 主色调按下
$successColor = [System.Drawing.Color]::FromArgb(40, 167, 69)       # 成功色
$successHover = [System.Drawing.Color]::FromArgb(33, 147, 58)       # 成功色悬停
$warningColor = [System.Drawing.Color]::FromArgb(255, 193, 7)       # 警告色
$warningHover = [System.Drawing.Color]::FromArgb(230, 173, 6)       # 警告色悬停
$infoColor = [System.Drawing.Color]::FromArgb(23, 162, 184)         # 信息色
$infoHover = [System.Drawing.Color]::FromArgb(19, 142, 164)         # 信息色悬停
$dangerColor = [System.Drawing.Color]::FromArgb(220, 53, 69)        # 危险色
$dangerHover = [System.Drawing.Color]::FromArgb(200, 35, 51)        # 危险色悬停
$secondaryColor = [System.Drawing.Color]::FromArgb(108, 117, 125)   # 次要色
$secondaryHover = [System.Drawing.Color]::FromArgb(92, 99, 106)     # 次要色悬停
$lightColor = [System.Drawing.Color]::FromArgb(248, 249, 250)       # 浅色背景
$darkColor = [System.Drawing.Color]::FromArgb(52, 58, 64)           # 深色背景
$textPrimary = [System.Drawing.Color]::FromArgb(33, 37, 41)         # 主要文字
$textSecondary = [System.Drawing.Color]::FromArgb(108, 117, 125)    # 次要文字

# =============================================
# 字体定义
# =============================================
$fontFamily = "微软雅黑"
$fontTitle = New-Object System.Drawing.Font($fontFamily, 14, [System.Drawing.FontStyle]::Bold)
$fontSubtitle = New-Object System.Drawing.Font($fontFamily, 10, [System.Drawing.FontStyle]::Regular)
$fontButton = New-Object System.Drawing.Font($fontFamily, 10, [System.Drawing.FontStyle]::Bold)
$fontButtonRegular = New-Object System.Drawing.Font($fontFamily, 10, [System.Drawing.FontStyle]::Regular)
$fontLabel = New-Object System.Drawing.Font($fontFamily, 9, [System.Drawing.FontStyle]::Regular)
$fontLog = New-Object System.Drawing.Font($fontFamily, 9, [System.Drawing.FontStyle]::Regular)

# =============================================
# 日志函数 - 修复版：不在控制台显示，只显示到软件界面
# =============================================
function Write-Log {
    param (
        [string]$Message,
        [string]$Type = "Info"  # Info, Success, Warning, Error
    )
    
    $timestamp = Get-Date -Format "HH:mm:ss"
    $logMessage = "[$timestamp] $Message"
    
    # 注意：移除了控制台输出，只输出到软件界面
    
    # 尝试写入文本框
    if ($global:logTextBox -ne $null -and $global:logTextBox.IsHandleCreated) {
        try {
            $global:logTextBox.Invoke([Action]{
                # 检查是否需要先添加换行
                $currentText = $global:logTextBox.Text
                if ($currentText.Length -gt 0) {
                    $lastChar = $currentText[-1]
                    if ($lastChar -ne "`n" -and $lastChar -ne "`r") {
                        $global:logTextBox.AppendText("`r`n")
                    }
                }
                
                $global:logTextBox.SelectionStart = $global:logTextBox.TextLength
                $global:logTextBox.SelectionLength = 0
                
                switch ($Type) {
                    "Success" { $global:logTextBox.SelectionColor = $successColor }
                    "Warning" { $global:logTextBox.SelectionColor = $warningColor }
                    "Error"   { $global:logTextBox.SelectionColor = $dangerColor }
                    default   { $global:logTextBox.SelectionColor = $textPrimary }
                }
                
                $global:logTextBox.AppendText("$logMessage`r`n")
                $global:logTextBox.SelectionColor = $global:logTextBox.ForeColor
                
                # 保持日志文件大小可控
                if ($global:logTextBox.Lines.Count -gt 1000) {
                    $lines = $global:logTextBox.Lines
                    $startIndex = [math]::Max(0, $lines.Count - 500)
                    $global:logTextBox.Text = [string]::Join("`r`n", $lines[$startIndex..($lines.Count-1)])
                }
                
                $global:logTextBox.ScrollToCaret()
            })
        }
        catch {
            # 如果文本框写入失败，静默处理（不在控制台输出）
        }
    }
}

# =============================================
# 进度显示函数
# =============================================
function Update-Progress {
    param (
        [int]$Value = -1,
        [string]$Text = "",
        [bool]$Marquee = $false
    )
    
    # 只在文本变化时更新日志
    if ($Text -ne $global:lastProgressText -and $Text -ne "") {
        $global:lastProgressText = $Text
        Write-Log $Text -Type "Info"
    }
    
    # 如果需要，更新进度条
    if ($global:progressBar -ne $null -and $global:progressBar.IsHandleCreated) {
        $global:progressBar.Invoke([Action]{
            if ($Marquee) {
                $global:progressBar.Style = 'Marquee'
            } else {
                $global:progressBar.Style = 'Continuous'
                if ($Value -ge 0 -and $Value -le 100) {
                    $global:progressBar.Value = $Value
                }
            }
        })
    }
}

function Hide-Progress {
    $global:lastProgressText = ""
}

# =============================================
# 高速下载函数
# =============================================
function Get-FileFromWeb {
    param (
        [Parameter(Mandatory)][string]$URL,
        [Parameter(Mandatory)][string]$File,
        [string]$DisplayName = "",
        [int]$MaxRetries = 3,
        [int]$BufferSize = 2097152
    )
    
    if ($DisplayName -eq "") {
        $DisplayName = [System.IO.Path]::GetFileName($URL)
    }
    
    # 优化网络设置
    [System.Net.ServicePointManager]::DefaultConnectionLimit = 100
    [System.Net.ServicePointManager]::Expect100Continue = $false
    [System.Net.ServicePointManager]::SecurityProtocol = [System.Net.SecurityProtocolType]::Tls12 -bor [System.Net.SecurityProtocolType]::Tls11 -bor [System.Net.SecurityProtocolType]::Tls
    
    $retryCount = 0
    $downloadSuccess = $false
    
    while ($retryCount -lt $MaxRetries -and -not $downloadSuccess) {
        $retryCount++
        
        if ($retryCount -gt 1) {
            Write-Log "第 $retryCount 次重试下载: $DisplayName" -Type "Warning"
            Start-Sleep -Seconds 2
        }
        
        try {
            # 创建目录
            $dir = [System.IO.Path]::GetDirectoryName($File)
            if (!(Test-Path $dir)) {
                New-Item -ItemType Directory -Path $dir -Force | Out-Null
            }
            
            Write-Log "开始下载: $DisplayName (尝试 $retryCount/$MaxRetries)" -Type "Info"
            
            # 创建请求
            $request = [System.Net.HttpWebRequest]::Create($URL)
            $request.Timeout = 30000
            $request.ReadWriteTimeout = 300000
            
            # 支持断点续传
            $startPosition = 0
            if (Test-Path $File) {
                $existingFile = Get-Item $File -ErrorAction SilentlyContinue
                if ($existingFile) {
                    $startPosition = $existingFile.Length
                    $request.AddRange($startPosition)
                    Write-Log "检测到已下载部分文件: $([math]::Round($startPosition / 1MB, 2)) MB" -Type "Info"
                }
            }
            
            Write-Log "正在连接到服务器..." -Type "Info"
            
            $response = $request.GetResponse()
            
            if ($response.StatusCode -eq 401 -or $response.StatusCode -eq 403 -or $response.StatusCode -eq 404) {
                throw "远程文件不存在、未授权或禁止访问：'$URL'。"
            }
            
            # 获取文件总大小
            [long]$fullSize = $response.ContentLength
            if ($startPosition -gt 0) {
                $fullSize += $startPosition
            }
            
            Write-Log "开始下载: $DisplayName (大小: $([math]::Round($fullSize / 1MB, 2)) MB)" -Type "Info"
            
            # 创建缓冲区
            [byte[]]$buffer = New-Object byte[] $BufferSize
            [long]$total = $startPosition
            [long]$count = 0
            
            $startTime = Get-Date
            $reader = $response.GetResponseStream()
            
            # 以追加模式打开文件
            $fileMode = if ($startPosition -gt 0) { 'Append' } else { 'Create' }
            $writer = New-Object System.IO.FileStream $File, $fileMode
            
            # 优化：减少UI更新频率
            $lastUpdateTime = Get-Date
            $lastUpdateSize = $total
            $updateIntervalMB = 10
            
            do {
                $count = $reader.Read($buffer, 0, $buffer.Length)
                $writer.Write($buffer, 0, $count)
                $total += $count
                
                # 计算是否需要更新UI
                $currentTime = Get-Date
                $downloadedSinceLastUpdate = $total - $lastUpdateSize
                $timeSinceLastUpdate = ($currentTime - $lastUpdateTime).TotalSeconds
                
                if ($downloadedSinceLastUpdate -ge (10 * 1024 * 1024) -or $timeSinceLastUpdate -ge 3) {
                    if ($fullSize -gt 0) {
                        $percent = $total / $fullSize
                        $percentComplete = $percent * 100
                        
                        $downloadedMB = [math]::Round($total / 1MB, 2)
                        $totalMB = [math]::Round($fullSize / 1MB, 2)
                        
                        # 计算瞬时速度
                        $speedMBps = 0
                        if ($timeSinceLastUpdate -gt 0) {
                            $speedMBps = [math]::Round($downloadedSinceLastUpdate / 1MB / $timeSinceLastUpdate, 2)
                        }
                        
                        # 计算剩余时间
                        $remainingTime = ""
                        if ($speedMBps -gt 0 -and $percentComplete -lt 100) {
                            $remainingMB = ($fullSize - $total) / 1MB
                            $remainingSeconds = [math]::Round($remainingMB / $speedMBps)
                            if ($remainingSeconds -gt 0) {
                                $remainingTime = " | 剩余: " + (New-TimeSpan -Seconds $remainingSeconds).ToString("hh\:mm\:ss")
                            }
                        }
                        
                        $progressText = "$DisplayName - $($percentComplete.ToString('##0.00').PadLeft(6))% ($downloadedMB MB / $totalMB MB) | 速度: $speedMBps MB/s$remainingTime"
                        Update-Progress -Value $percentComplete -Text $progressText
                        
                        $lastUpdateTime = $currentTime
                        $lastUpdateSize = $total
                    }
                }
                
            } while ($count -gt 0)
            
            $writer.Close()
            $reader.Close()
            $response.Close()
            
            $endTime = Get-Date
            $downloadTime = ($endTime - $startTime).TotalSeconds
            
            if ($downloadTime -gt 0) {
                $downloadSpeed = [math]::Round(($fullSize / $downloadTime) / 1MB, 2)
                Write-Log "✓ 下载完成: $DisplayName" -Type "Success"
                Write-Log "  下载时间: $downloadTime 秒 | 平均速度: $downloadSpeed MB/s | 文件大小: $([math]::Round($fullSize / 1MB, 2)) MB" -Type "Info"
            } else {
                Write-Log "✓ 下载完成: $DisplayName" -Type "Success"
            }
            
            # 验证文件完整性
            if (Test-Path $File) {
                $downloadedFile = Get-Item $File
                if ($downloadedFile.Length -eq $fullSize) {
                    Write-Log "✓ 文件完整性验证通过" -Type "Success"
                    $downloadSuccess = $true
                } else {
                    Write-Log "! 文件大小不匹配，可能下载不完整" -Type "Warning"
                    Write-Log "  期望: $fullSize 字节, 实际: $($downloadedFile.Length) 字节" -Type "Info"
                    
                    if ($retryCount -lt $MaxRetries) {
                        Remove-Item $File -Force -ErrorAction SilentlyContinue
                        continue
                    } else {
                        throw "下载文件不完整，已尝试 $MaxRetries 次"
                    }
                }
            }
            
            Hide-Progress
            return $true
        }
        catch {
            Write-Log "✗ 下载失败 (尝试 $retryCount/$MaxRetries): $($_.Exception.Message)" -Type "Error"
            
            # 清理资源
            if ($writer -ne $null) { try { $writer.Close() } catch {} }
            if ($reader -ne $null) { try { $reader.Close() } catch {} }
            if ($response -ne $null) { try { $response.Close() } catch {} }
            
            if ($retryCount -eq $MaxRetries) {
                Write-Log "✗ 下载失败，已尝试 $MaxRetries 次" -Type "Error"
                Hide-Progress
                return $false
            }
        }
    }
    
    Hide-Progress
    return $false
}

# =============================================
# 通用功能函数
# =============================================

function Show-Info {
    param (
        [string]$Version = "2023"  # 2023 或 2019
    )
    
    if ($Version -eq "2023") {
        $info = @"
WPS Office 2023 专业增强版 - V12.8.2.21555

版本说明：
WPS2023专业增强版：免激活、去水印、永久授权、完整功能优化增强版
1. 基于官方WPS2023专业版打包，自动调用安装脚本写入授权序列号
2. 集成VBA组件、终身授权序列号、安装完毕即WPS激活专业增强版
3. 去广告优化、去我的电脑WPS云盘、保留账户登陆和云同步上传下载
   ﹂去界面左侧：日历、WPS便签、会议、统计表单 (广告)
   ﹂去应用中心：分享协作功能网页入口（会议、统计表单）
   ﹂彻底去升级：无版本更新提示，检查更新永远都是最新版
4. 安装过程自动剔除桌面和我的电脑WPS云盘虚拟盘符入口
5. 安装过程自动禁用WPS办公助手和其资源管理器右键菜单
6. 安装过程自动删除升级组件并移除检查升级的计划任务项
7. 安装后删除升级文件，彻底禁用更新
"@
        Write-Log "========== WPS Office 2023 版本信息 ==========" -Type "Info"
    } else {
        $info = @"
WPS Office 2019 专业增强版 - V11.8.2.12300

版本说明：
WPS2019专业增强版：免激活、去水印、永久授权、完整功能优化增强版
1. 基于官方WPS2019专业版打包，自动调用安装脚本写入授权序列号
2. 集成VBA组件、授权序列号、安装完毕即WPS激活专业增强版
3. 去广告优化、去我的电脑WPS云盘、保留账户登陆和云同步上传下载
   ﹂去界面左侧：日历、WPS便签、会议、统计表单 (广告)
   ﹂去应用中心：分享协作功能网页入口（会议、统计表单）
   ﹂彻底去升级：无版本更新提示，检查更新永远都是最新版
4. 安装过程自动剔除桌面和我的电脑WPS云盘虚拟盘符入口
5. 安装过程自动删除升级组件并移除检查升级的计划任务项
6. 安装后删除升级文件，彻底禁用更新
"@
        Write-Log "========== WPS Office 2019 版本信息 ==========" -Type "Info"
    }
    
    $infoLines = $info -split "`n"
    foreach ($line in $infoLines) {
        Write-Log $line -Type "Info"
    }
}

function Check-WPSInstalled {
    param(
        [switch]$Detailed = $false  # 是否输出详细信息
    )
    
    Write-Log "检查系统中是否已安装WPS Office..." -Type "Info"
    
    $found = $false
    $installInfo = @{
        IsInstalled = $false
        Path = ""
        Version = ""
        DisplayName = ""
        RegistryEntries = @()
        RegistryCount = 0
    }
    
    # 检查安装目录
    $paths = @(
        "${env:ProgramFiles}\WPS Office",
        "${env:ProgramFiles(x86)}\WPS Office",
        "${env:ProgramFiles}\Kingsoft\WPS Office",
        "${env:ProgramFiles(x86)}\Kingsoft\WPS Office"
    )
    
    foreach ($path in $paths) {
        if (Test-Path $path) {
            $found = $true
            $installInfo.IsInstalled = $true
            $installInfo.Path = $path
            
            $versionFile = Join-Path $path "office6\cfgs\oem.ini"
            if (Test-Path $versionFile) {
                $versionInfo = Get-Content $versionFile -Encoding Default -ErrorAction SilentlyContinue | 
                              Where-Object { $_ -match "ProductVersion" }
                if ($versionInfo) {
                    $version = $versionInfo.Split('=')[1].Trim()
                    $installInfo.Version = $version
                    break
                }
            }
            break
        }
    }
    
    # 搜索所有可能的WPS相关注册表项
    $wpsKeys = @()
    $regPaths = @(
        "HKLM:\SOFTWARE\Microsoft\Windows\CurrentVersion\Uninstall",
        "HKLM:\SOFTWARE\WOW6432Node\Microsoft\Windows\CurrentVersion\Uninstall",
        "HKCU:\SOFTWARE\Microsoft\Windows\CurrentVersion\Uninstall"
    )
    
    foreach ($path in $regPaths) {
        if (Test-Path $path) {
            try {
                $keys = Get-ChildItem $path -ErrorAction SilentlyContinue | 
                    ForEach-Object { 
                        try {
                            Get-ItemProperty $_.PSPath -ErrorAction SilentlyContinue
                        } catch {
                            # 跳过无法访问的注册表项
                        }
                    } | 
                    Where-Object { 
                        $_.DisplayName -like "*WPS*" -or 
                        $_.DisplayName -like "*Kingsoft*" -or
                        $_.DisplayName -like "*金山*"
                    }
                
                if ($keys) {
                    $wpsKeys += $keys
                    $found = $true
                    $installInfo.IsInstalled = $true
                }
            }
            catch {
                # 静默处理注册表访问错误
            }
        }
    }
    
    # 整理注册表信息
    if ($wpsKeys.Count -gt 0) {
        $installInfo.RegistryEntries = $wpsKeys | ForEach-Object {
            @{
                DisplayName = $_.DisplayName
                DisplayVersion = $_.DisplayVersion
                InstallLocation = $_.InstallLocation
                Publisher = $_.Publisher
                UninstallString = $_.UninstallString
            }
        }
        $installInfo.RegistryCount = $wpsKeys.Count
        
        # 如果没有找到目录信息，使用第一个注册表项的安装位置
        if ($installInfo.Path -eq "" -and $wpsKeys[0].InstallLocation) {
            $installInfo.Path = $wpsKeys[0].InstallLocation
        }
        if ($installInfo.Version -eq "" -and $wpsKeys[0].DisplayVersion) {
            $installInfo.Version = $wpsKeys[0].DisplayVersion
        }
        if ($installInfo.DisplayName -eq "" -and $wpsKeys[0].DisplayName) {
            $installInfo.DisplayName = $wpsKeys[0].DisplayName
        }
    }
    
    if ($found) {
        if (-not $Detailed) {
            # 简洁模式：只在安装流程中使用
            if ($installInfo.Path -ne "") {
                Write-Log "✓ 发现已安装的WPS Office" -Type "Warning"
                Write-Log "  安装目录: $($installInfo.Path)" -Type "Info"
            } else {
                Write-Log "✓ 发现已安装的WPS Office (位置未知)" -Type "Warning"
            }
        } else {
            # 详细模式：简化输出，只显示核心信息
            Write-Log "=== WPS Office 详细安装信息 ===" -Type "Info"
            
            if ($installInfo.Path -ne "") {
                Write-Log "安装目录: $($installInfo.Path)" -Type "Info"
            }
            
            if ($installInfo.DisplayName -ne "") {
                Write-Log "名称: $($installInfo.DisplayName)" -Type "Info"
            }
            
            if ($installInfo.Version -ne "") {
                Write-Log "版本: $($installInfo.Version)" -Type "Info"
            }
            
            # 显示注册表项数量，但不再显示每个项的详细内容
            if ($installInfo.RegistryCount -gt 0) {
                Write-Log "找到的注册表项 ($($installInfo.RegistryCount)个):" -Type "Info"
            }
            
            Write-Log "如需重新安装，请先通过控制面板卸载" -Type "Warning"
        }
        
        return $installInfo
    } else {
        Write-Log "未检测到已安装的WPS Office" -Type "Success"
        return @{ IsInstalled = $false }
    }
}

function Clean-LicenseFile {
    Write-Log "清理许可证文件..." -Type "Info"
    
    $licensePath = "C:\ProgramData\Kingsoft\office6\license2.dat"
    
    if (Test-Path $licensePath) {
        try {
            # 移除只读、隐藏属性（已注释，仅保留备注）
            # $file = Get-Item $licensePath -Force
            # $file.Attributes = $file.Attributes -band (-bnot [System.IO.FileAttributes]::ReadOnly)
            # $file.Attributes = $file.Attributes -band (-bnot [System.IO.FileAttributes]::Hidden)
            
            # 删除文件
            Remove-Item $licensePath -Force -ErrorAction Stop
            Write-Log "✓ 已删除旧的许可证文件" -Type "Success"
        }
        catch {
            Write-Log "! 删除许可证文件失败: $($_.Exception.Message)" -Type "Warning"
        }
    } else {
        Write-Log "未找到许可证文件，无需清理" -Type "Info"
    }
}

function Generate-ConfigFile {
    param (
        [string]$ConfigPath,
        [string]$Version = "2023"  # 2023 或 2019
    )
    
    if ($Version -eq "2023") {
        $configContent = @'
[support]
OCRTool=true
XiuTang=false
WeiboPlugin=true
EnablePdf2WordV2=true
EnableProcessonMind=true
EnableProcessonFlow=true
IsSupportPhoto2pdf=true
EnableProcessonFlow/Mind=true
WPSPlusVersion=true
NseVisible=true
Update=false
IntranetDrive=false
EnterpriseDocpermission=true
EnablePlainWatermarkInfo=true
Prometheus=false
DisableWPSPdfDeskTopShortcut=true
OnlineWithoutCloudDoc=false
WeChatCustomerService=false
EnableWPSOfficialWebsite=false
UpdateSvrCustomAddress=http://0.0.0.0/wpsoffice/updateserver/update

[Setup]
SendInstallSource=1
Silent=1
Sn=TJ3GN-9NTGQ-GLF7C-YEN8X-TJWML
SourceDir=oeminfo
UpdateSvrDefaultAddress=http://127.0.0.1

[Auth]
LicenseServerUrl=http://wps.mycustom.server/account
KSNUpdateServerUrl=http://0.0.0.0

[Server]
UpdateServer=http://127.0.0.1
InfoCollectServer=http://0.0.0.0/wpsv6internet/infos.ads
IntranetPluginsICServer=http://0.0.0.0/xva/v1/plugin/stat
CustomUpdateServer=http://0.0.0.0/wpsoffice/updateserver/update
'@
    } else {
        # 2019版本配置
        $configContent = @'
[support]
OCRTool=true
XiuTang=false
WeiboPlugin=true
EnablePdf2WordV2=true
EnableProcessonMind=true
EnableProcessonFlow=true
IsSupportPhoto2pdf=true
WPSPlusVersion=true ;显示增强版
NseVisible=true  ;安装VB组件工具
Update=false     ;禁止版本检测升级
IntranetDrive=true ;不显示我的电脑WPS云盘
EnterpriseDocpermission=true ;启用企业版文档权限
EnablePlainWatermarkInfo=true ;输出打印无水印信息
Prometheus=false ;禁用整合模式(打开应用需要手动新建文件)
IsCreateNewFile=1 ; 整合模式实现（打开应用即进入编辑模式）
Support2016SN=true ;启用序列号(11.8.2.12287 版开始已失效)
DisableWPSPdfDeskTopShortcut=true ;不创建桌面PDF快捷方式
FileDialogDefWpsCloud=false  ;保存对话框可自定义WPS云文档
OnlineWithoutCloudDoc=false  ;不显示广告 (日历, WPS便签, 会议, 统计表单)
WeChatCustomerService=false  ;不显示工具栏WPS客服微信按钮
UpdateSvrCustomAddress=http://0.0.0.0/wpsoffice/updateserver/update
caSM7oYrI4vjjbzpO04RdA..=NsbhfV4nLv_oZGENyLSVZA..
VaRJdz9TqngvP2rCk3bf-dH_HWpeBrG_9hpfdL1oM7Y.=NsbhfV4nLv_oZGENyLSVZA..  ;不显示工具栏WPS客服微信按钮 (最新版)
XAoSmN4Dk8PJG50puLaNe-iIrz2EFPJoaiSwWCbv1_A.=NsbhfV4nLv_oZGENyLSVZA..  ;不显示广告 (日历, WPS便签, 会议, 统计表单)(最新版)
uHggomDKEaVGML_Zzoprjw..=NsbhfV4nLv_oZGENyLSVZA..
PZ8bJ6Ee7I1QgyE_ecJvbiWJqbfBIQrcAzfOWLgeT1c.=WHfH10HHgeQrW2N48LfXrA..
jko0IwcTvUkzCbPvis6rUIxF_x6CEcMTNTiM8kp_W9A.=NsbhfV4nLv_oZGENyLSVZA..

[Setup]
SendInstallSource=1 ;检测安装包 
Silent=0 ;0带界面标准安装，1无界面静默安装
Sn=TJ3GN-9NTGQ-GLF7C-YEN8X-TJWML ;终身授权序列号 
SourceDir=oeminfo ;源目录支持读取企业OEM配置文件安装
UpdateSvrDefaultAddress=http://127.0.0.1 ;升级服务默认服务器
Kr0cNR1nKpC7yGiK-_5E2w..=yrogMsuo-CgeSd6yrWZsOBiJUkFphK4lPxYTSyyRgqc.

[Auth]
;自定义授权序列号验证和升级服务器
LicenseServerUrl=http://127.0.0.1
KSNUpdateServerUrl=http://0.0.0.0

[Server]
UpdateServer=http://127.0.0.1
CustomUpdateServer=http://0.0.0.0
AutoNomInfoSvr=http://0.0.0.0/wpscollect
QingUrl=http://0.0.0.0
'@
    }
    
    $configContent | Out-File -FilePath $ConfigPath -Encoding Default
    Write-Log "✓ $Version 版本配置文件已生成" -Type "Success"
}

# =============================================
# 核心修复：安全的路径处理函数
# =============================================
function Safe-TrimPath {
    param ([string]$Path)
    
    # 如果路径为空或null，返回默认路径
    if ([string]::IsNullOrWhiteSpace($Path)) {
        return "${env:ProgramFiles(x86)}\Kingsoft\WPS Office"
    }
    
    # 确保是字符串
    $Path = [string]$Path
    
    # 安全地调用Trim
    try {
        return $Path.Trim()
    }
    catch {
        # 如果Trim失败，返回原字符串
        return $Path
    }
}

# =============================================
# 改进的查找WPS目录（解决子目录查找问题）
# =============================================
function Find-WPSActualPath {
    param ([string]$BasePath)
    
    Write-Log "查找WPS实际安装目录..." -Type "Info"
    
    # 安全处理路径
    $BasePath = Safe-TrimPath -Path $BasePath
    
    # 1. 先检查直接路径
    $ksomiscPath = Join-Path $BasePath "office6\ksomisc.exe"
    if (Test-Path $ksomiscPath) {
        return $BasePath
    }
    
    # 2. 检查子目录（改进版本）
    if (Test-Path $BasePath) {
        try {
            # 获取所有子目录
            $subDirs = @()
            try {
                $subDirs = Get-ChildItem -Path $BasePath -Directory -ErrorAction Stop
            }
            catch {
                return $BasePath
            }
            
            # 检查每个子目录
            foreach ($subDir in $subDirs) {
                $subPath = $subDir.FullName
                $subKsomiscPath = Join-Path $subPath "office6\ksomisc.exe"
                
                if (Test-Path $subKsomiscPath) {
                    return $subPath
                }
            }
            
            # 如果没找到，检查子目录的子目录（嵌套查找）
            foreach ($subDir in $subDirs) {
                $subPath = $subDir.FullName
                $nestedSubDirs = Get-ChildItem -Path $subPath -Directory -ErrorAction SilentlyContinue
                
                foreach ($nestedDir in $nestedSubDirs) {
                    $nestedPath = $nestedDir.FullName
                    $nestedKsomiscPath = Join-Path $nestedPath "office6\ksomisc.exe"
                    
                    if (Test-Path $nestedKsomiscPath) {
                        return $nestedPath
                    }
                }
            }
            
            # 查找任何包含office6的目录
            foreach ($subDir in $subDirs) {
                $subPath = $subDir.FullName
                $office6Path = Join-Path $subPath "office6"
                
                if (Test-Path $office6Path) {
                    return $subPath
                }
            }
        }
        catch {
            Write-Log "! 检查子目录失败: $($_.Exception.Message)" -Type "Warning"
        }
    } else {
        Write-Log "! 基目录不存在: $BasePath" -Type "Warning"
    }
    
    Write-Log "! 未找到注册程序，使用基目录" -Type "Warning"
    return $BasePath
}

# =============================================
# 完全修复的安装后处理（简化版）
# =============================================
function Post-InstallProcessing {
    param ([string]$InstallPath)
    
    Write-Log "开始安装后处理..." -Type "Info"
    
    $activated = $false  # 记录激活状态
    
    try {
        # 1. 安全处理安装路径
        $safePath = Safe-TrimPath -Path $InstallPath
        
        # 2. 查找实际目录
        $actualPath = Find-WPSActualPath -BasePath $safePath
        $actualPath = Safe-TrimPath -Path $actualPath
        
        # 3. 执行序列号注册
        Write-Log "执行序列号注册..." -Type "Info"
        $ksomiscPath = Join-Path $actualPath "office6\ksomisc.exe"
        
        # 检查许可证文件是否已存在
        $licensePath = "C:\ProgramData\Kingsoft\office6\license2.dat"
        $licenseExists = Test-Path $licensePath
        
        if (Test-Path $ksomiscPath) {
            # 如果许可证文件已存在，删除旧文件
            if ($licenseExists) {
                try {
                    Remove-Item -Path $licensePath -Force -ErrorAction SilentlyContinue
                }
                catch {}
            }
            
            try {
                # 执行命令
                Write-Log "执行注册命令" -Type "Info"
                
                $process = Start-Process -FilePath $ksomiscPath -ArgumentList "-addsn TJ3GN-9NTGQ-GLF7C-YEN8X-TJWML" -Wait -NoNewWindow -PassThru
                
                # 等待命令执行完成
                Start-Sleep -Seconds 3
                
                # 检查退出代码并验证许可证文件
                if (Test-Path $licensePath) {
                    Write-Log "✓ 序列号注册成功 - 许可证文件已创建" -Type "Success"
                    
                    # 设置只读和隐藏属性（已注释，仅保留备注）
                    # try {
                    #     $file = Get-Item $licensePath -Force
                    #     $file.Attributes = $file.Attributes -bor [System.IO.FileAttributes]::ReadOnly
                    #     $file.Attributes = $file.Attributes -bor [System.IO.FileAttributes]::Hidden
                    #     Write-Log "✓ 已设置许可证文件为只读和隐藏" -Type "Success"
                    # }
                    # catch {}
                    $activated = $true
                } else {
                    Write-Log "! 序列号注册失败 - 许可证文件未创建" -Type "Warning"
                }
            }
            catch {
                Write-Log "! 执行注册命令失败" -Type "Warning"
            }
        } else {
            Write-Log "! 未找到注册文件" -Type "Warning"
        }
        
        # 4. 简化更新禁用：直接删除wpsupdate.exe并创建空的只读文件
        Write-Log "禁用更新功能..." -Type "Info"
        
        # 可能的wpsupdate.exe路径
        $possiblePaths = @()
        $possiblePaths += Join-Path $actualPath "wtoolex\wpsupdate.exe"
        $possiblePaths += Join-Path $actualPath "wpsupdate.exe"
        $possiblePaths += Join-Path $actualPath "office6\wpsupdate.exe"
        $possiblePaths += Join-Path $actualPath "office6\wtoolex\wpsupdate.exe"
        
        $foundAndDisabled = $false
        
        foreach ($updatePath in $possiblePaths) {
            if (Test-Path $updatePath) {
                Write-Log "找到更新文件" -Type "Info"
                
                try {
                    # 直接删除原始文件
                    Remove-Item -Path $updatePath -Force -ErrorAction Stop
                    Write-Log "✓ 已删除更新文件" -Type "Success"
                    
                    # 创建空的只读文件
                    try {
                        # 创建空的文件
                        "" | Out-File -FilePath $updatePath -Encoding ASCII -Force
                        
                        # 设置文件属性为只读和隐藏
                        $file = Get-Item $updatePath -Force
                        $file.Attributes = $file.Attributes -bor [System.IO.FileAttributes]::ReadOnly
                        $file.Attributes = $file.Attributes -bor [System.IO.FileAttributes]::Hidden
                        
                        Write-Log "✓ 已创建空的只读占位文件" -Type "Success"
                        $foundAndDisabled = $true
                        break
                    }
                    catch {
                        Write-Log "! 创建占位文件失败" -Type "Warning"
                    }
                }
                catch {
                    Write-Log "! 处理更新文件失败" -Type "Warning"
                }
            }
        }
        
        if (-not $foundAndDisabled) {
            Write-Log "! 未找到更新文件" -Type "Warning"
        } else {
            Write-Log "✓ 更新功能已禁用" -Type "Success"
        }
        
        # 根据处理结果输出相应消息
        if ($activated) {
            Write-Log "✓ WPS Office 已成功激活" -Type "Success"
            Write-Log "序列号已写入" -Type "Success"
            Write-Log "提示: 请重启WPS Office以完成激活" -Type "Info"
            return $true
        } else {
            Write-Log "! WPS激活失败，请手动激活" -Type "Warning"
            return $false
        }
    }
    catch {
        Write-Log "✗ 安装后处理出错" -Type "Error"
        if ($activated) {
            Write-Log "✓ WPS Office 已成功激活" -Type "Success"
        } else {
            Write-Log "WPS已安装完成，但需要手动激活" -Type "Info"
        }
        return $activated
    }
}

# =============================================
# 安装函数（优化日志输出）
# =============================================
function Install-WPS {
    param (
        [string]$Version = "2023"  # 2023 或 2019
    )
    
    # 检查是否正在安装
    if ($global:isInstalling) {
        Write-Log "安装正在进行中，请稍候..." -Type "Warning"
        return $false
    }
    
    $global:isInstalling = $true
    
    # 更新按钮状态
    if ($global:installButton -ne $null) {
        $global:installButton.Invoke([Action]{
            $global:installButton.Enabled = $false
            $global:installButton.Text = "安装中..."
            $global:installButton.BackColor = $secondaryColor
        })
    }
    if ($global:install2019Button -ne $null) {
        $global:install2019Button.Invoke([Action]{
            $global:install2019Button.Enabled = $false
            $global:install2019Button.Text = "安装中..."
            $global:install2019Button.BackColor = $secondaryColor
        })
    }
    [System.Windows.Forms.Application]::DoEvents()
    
    try {
        Write-Log "========== 开始安装 WPS Office $Version ==========" -Type "Info"
        
        # 1. 检查是否已安装（使用简洁模式）
        $installInfo = Check-WPSInstalled
        
        if ($installInfo.IsInstalled) {
            Write-Log "✗ 系统已安装WPS Office，安装已取消" -Type "Error"
            Write-Log "如需重新安装，请先通过控制面板卸载现有版本" -Type "Info"
            
            $global:isInstalling = $false
            if ($global:installButton -ne $null) {
                $global:installButton.Invoke([Action]{
                    $global:installButton.Enabled = $true
                    $global:installButton.Text = "安装 WPS 2023"
                    $global:installButton.BackColor = $primaryColor
                })
            }
            if ($global:install2019Button -ne $null) {
                $global:install2019Button.Invoke([Action]{
                    $global:install2019Button.Enabled = $true
                    $global:install2019Button.Text = "安装 WPS 2019"
                    $global:install2019Button.BackColor = $secondaryColor
                })
            }
            [System.Windows.Forms.Application]::DoEvents()
            return $false
        }
        
        Write-Log "✓ 系统未安装WPS Office，可以继续安装" -Type "Success"
        
        # 2. 清理旧的许可证文件
        Clean-LicenseFile
        
        # 3. 创建临时目录
        Write-Log "创建临时目录..." -Type "Info"
        $tempDir = Join-Path $env:TEMP "WPS${Version}_Install_$(Get-Date -Format 'yyyyMMdd_HHmmss')"
        if (!(Test-Path $tempDir)) {
            New-Item -ItemType Directory -Path $tempDir -Force | Out-Null
            Write-Log "✓ 临时目录已创建" -Type "Success"
        }
        
        # 4. 下载WPS安装包
        Write-Log "下载WPS安装包..." -Type "Info"
        
        # 根据版本选择下载URL
        if ($Version -eq "2023") {
            $wpsUrls = @(
                "http://192.168.8.251/api/soft/Wps/Wps2023/WPS_Setup.exe",
                "https://tool.3721611.com/api/soft/Wps/Wps2023/WPS_Setup.exe"
            )
            $displayName = "WPS Office 2023 安装包"
        } else {
            $wpsUrls = @(
                "http://192.168.8.251/api/soft/Wps/Wps2019/WPS_Setup.exe",
                "https://tool.3721611.com/api/soft/Wps/Wps2019/WPS_Setup.exe"
            )
            $displayName = "WPS Office 2019 安装包"
        }
        
        $wpsPath = Join-Path $tempDir "WPS_Setup.exe"
        $wpsDownloaded = $false
        
        # 尝试多个下载源
        foreach ($wpsUrl in $wpsUrls) {
            try {
                $wpsDownloaded = Get-FileFromWeb -URL $wpsUrl -File $wpsPath -DisplayName $displayName
                if ($wpsDownloaded) {
                    break
                }
            }
            catch {
                Write-Log "下载源失败" -Type "Warning"
            }
        }
        
        if (-not $wpsDownloaded) {
            Write-Log "✗ WPS安装包下载失败，安装中止" -Type "Error"
            $global:isInstalling = $false
            if ($global:installButton -ne $null) {
                $global:installButton.Invoke([Action]{
                    $global:installButton.Enabled = $true
                    $global:installButton.Text = "安装 WPS 2023"
                    $global:installButton.BackColor = $primaryColor
                })
            }
            if ($global:install2019Button -ne $null) {
                $global:install2019Button.Invoke([Action]{
                    $global:install2019Button.Enabled = $true
                    $global:install2019Button.Text = "安装 WPS 2019"
                    $global:install2019Button.BackColor = $secondaryColor
                })
            }
            [System.Windows.Forms.Application]::DoEvents()
            return $false
        }
        
        # 5. 下载VBA组件（只下载，不安装，主安装包附件）
        Write-Log "下载VBA组件（附件）..." -Type "Info"
        
        if ($Version -eq "2023") {
            $vbaUrls = @(
                "http://192.168.8.251/api/soft/Wps/Wps2023/VBA_Setup.exe",
                "https://tool.3721611.com//api/soft/Wps/Wps2023/VBA_Setup.exe"
            )
        } else {
            $vbaUrls = @(
                "http://192.168.8.251/api/soft/Wps/Wps2019/VBA_Setup.exe",
                "https://tool.3721611.com//api/soft/Wps/Wps2019/VBA_Setup.exe"
            )
        }
        
        $vbaPath = Join-Path $tempDir "VBA_Setup.exe"
        $vbaDownloaded = $false
        
        foreach ($vbaUrl in $vbaUrls) {
            try {
                $vbaDownloaded = Get-FileFromWeb -URL $vbaUrl -File $vbaPath -DisplayName "VBA组件（附件）"
                if ($vbaDownloaded) {
                    break
                }
            }
            catch {
                Write-Log "VBA下载源失败" -Type "Warning"
            }
        }
        
        if (-not $vbaDownloaded) {
            Write-Log "! VBA组件下载失败" -Type "Warning"
        } else {
            Write-Log "✓ VBA组件下载成功（附件）" -Type "Success"
        }
        
        # 6. 生成配置文件
        Write-Log "生成配置文件..." -Type "Info"
        $configPath = Join-Path $tempDir "oem.ini"
        Generate-ConfigFile -ConfigPath $configPath -Version $Version
        
        # 7. 将配置文件与安装程序放在一起
        Write-Log "准备安装文件..." -Type "Info"
        $installDir = Split-Path $wpsPath -Parent
        $destConfigPath = Join-Path $installDir "oem.ini"
        
        if ($configPath -ne $destConfigPath) {
            Copy-Item -Path $configPath -Destination $destConfigPath -Force
            Write-Log "✓ 配置文件已放置到安装目录" -Type "Success"
        }
        
        # 8. 安装WPS（静默安装）
        Write-Log "安装WPS Office $Version..." -Type "Info"
        
        if (-not (Test-Path $wpsPath)) {
            Write-Log "✗ WPS安装文件不存在" -Type "Error"
            $global:isInstalling = $false
            if ($global:installButton -ne $null) {
                $global:installButton.Invoke([Action]{
                    $global:installButton.Enabled = $true
                    $global:installButton.Text = "安装 WPS 2023"
                    $global:installButton.BackColor = $primaryColor
                })
            }
            if ($global:install2019Button -ne $null) {
                $global:install2019Button.Invoke([Action]{
                    $global:install2019Button.Enabled = $true
                    $global:install2019Button.Text = "安装 WPS 2019"
                    $global:install2019Button.BackColor = $secondaryColor
                })
            }
            [System.Windows.Forms.Application]::DoEvents()
            return $false
        }
        
        # 使用静默安装
        $installArgs = "/S"
        
        $process = Start-Process -FilePath $wpsPath -ArgumentList $installArgs -PassThru -WindowStyle Hidden
        
        if ($process) {
            Write-Log "安装程序已启动" -Type "Info"
            
            # 等待安装完成
            $startTime = Get-Date
            $timeoutMinutes = 15
            $lastUpdate = $startTime
            $progress = 0
            
            while (-not $process.HasExited) {
                $elapsed = (Get-Date) - $startTime
                $minutes = [math]::Round($elapsed.TotalMinutes, 1)
                $seconds = [math]::Round($elapsed.TotalSeconds)
                
                if ($minutes -gt $timeoutMinutes) {
                    Write-Log "! 安装超时，正在终止进程..." -Type "Warning"
                    try {
                        Stop-Process -Id $process.Id -Force -ErrorAction SilentlyContinue
                    } catch {}
                    break
                }
                
                # 模拟进度更新
                $progress = [math]::Min(90, $minutes * 20)
                Update-Progress -Value $progress -Text "安装进度: 安装中... 已用时 $minutes 分钟"
                
                # 每10秒更新一次安装状态
                $currentTime = Get-Date
                if (($currentTime - $lastUpdate).TotalSeconds -ge 10) {
                    $lastUpdate = $currentTime
                }
                
                [System.Windows.Forms.Application]::DoEvents()
                Start-Sleep -Milliseconds 500
            }
            
            Update-Progress -Value 100 -Text "安装进度: 安装完成"
            
            # 检查退出代码
            if ($process.HasExited) {
                if ($process.ExitCode -eq 0) {
                    Write-Log "✓ WPS Office $Version 安装成功" -Type "Success"
                } else {
                    Write-Log "✗ WPS安装失败 (退出代码: $($process.ExitCode))" -Type "Error"
                    $global:isInstalling = $false
                    if ($global:installButton -ne $null) {
                        $global:installButton.Invoke([Action]{
                            $global:installButton.Enabled = $true
                            $global:installButton.Text = "安装 WPS 2023"
                            $global:installButton.BackColor = $primaryColor
                        })
                    }
                    if ($global:install2019Button -ne $null) {
                        $global:install2019Button.Invoke([Action]{
                            $global:install2019Button.Enabled = $true
                            $global:install2019Button.Text = "安装 WPS 2019"
                            $global:install2019Button.BackColor = $secondaryColor
                        })
                    }
                    [System.Windows.Forms.Application]::DoEvents()
                    return $false
                }
            }
        } else {
            Write-Log "✗ 无法启动安装程序" -Type "Error"
            $global:isInstalling = $false
            if ($global:installButton -ne $null) {
                $global:installButton.Invoke([Action]{
                    $global:installButton.Enabled = $true
                    $global:installButton.Text = "安装 WPS 2023"
                    $global:installButton.BackColor = $primaryColor
                })
            }
            if ($global:install2019Button -ne $null) {
                $global:install2019Button.Invoke([Action]{
                    $global:install2019Button.Enabled = $true
                    $global:install2019Button.Text = "安装 WPS 2019"
                    $global:install2019Button.BackColor = $secondaryColor
                })
            }
            [System.Windows.Forms.Application]::DoEvents()
            return $false
        }
        
        # 9. 等待安装完成并验证
        Write-Log "验证安装结果..." -Type "Info"
        Start-Sleep -Seconds 5
        
        $installSuccess = $false
        $actualInstallPath = ""
        
        # 查找实际安装路径
        $defaultPaths = @(
            "${env:ProgramFiles(x86)}\Kingsoft\WPS Office",
            "${env:ProgramFiles}\Kingsoft\WPS Office",
            "${env:ProgramFiles(x86)}\WPS Office",
            "${env:ProgramFiles}\WPS Office"
        )
        
        foreach ($path in $defaultPaths) {
            if (Test-Path $path) {
                $actualInstallPath = $path
                break
            }
        }
        
        if ($actualInstallPath -ne "" -and (Test-Path $actualInstallPath)) {
            $installSuccess = $true
            Write-Log "✓ WPS安装验证成功" -Type "Success"
            Write-Log "安装目录: $actualInstallPath" -Type "Info"
            
            # 10. 执行安装后处理
            Write-Log "执行安装后处理..." -Type "Info"
            $postProcessResult = Post-InstallProcessing -InstallPath $actualInstallPath
            
            if ($postProcessResult) {
                Write-Log "✓ 安装后处理完成" -Type "Success"
            } else {
                Write-Log "! 安装后处理部分失败" -Type "Warning"
            }
        } else {
            Write-Log "✗ 无法确定WPS安装目录" -Type "Error"
            $installSuccess = $false
        }
        
        # 11. 清理临时文件
        Write-Log "清理临时文件..." -Type "Info"
        try {
            if (Test-Path $tempDir) {
                Remove-Item -Path $tempDir -Recurse -Force -ErrorAction SilentlyContinue
                Write-Log "✓ 临时文件已清理" -Type "Success"
            }
        }
        catch {
            Write-Log "! 清理临时文件失败" -Type "Warning"
        }
        
        if ($installSuccess) {
            Write-Log "========== WPS Office $Version 安装完成！ ==========" -Type "Success"
            Write-Log "序列号已注册成功" -Type "Success"
            Write-Log "提示: 更新功能已禁用" -Type "Info"
            Write-Log "VBA组件由WPS主程序自动安装" -Type "Info"
            Write-Log "请重启WPS Office以完成激活" -Type "Info"
        } else {
            Write-Log "✗ WPS安装验证失败，请检查日志" -Type "Error"
        }
        
        $global:isInstalling = $false
        if ($global:installButton -ne $null) {
            $global:installButton.Invoke([Action]{
                $global:installButton.Enabled = $true
                $global:installButton.Text = "安装 WPS 2023"
                $global:installButton.BackColor = $primaryColor
            })
        }
        if ($global:install2019Button -ne $null) {
            $global:install2019Button.Invoke([Action]{
                $global:install2019Button.Enabled = $true
                $global:install2019Button.Text = "安装 WPS 2019"
                $global:install2019Button.BackColor = $secondaryColor
            })
        }
        [System.Windows.Forms.Application]::DoEvents()
        Hide-Progress
        
        return $installSuccess
    }
    catch {
        Write-Log "✗ 安装过程中发生错误" -Type "Error"
        $global:isInstalling = $false
        if ($global:installButton -ne $null) {
            $global:installButton.Invoke([Action]{
                $global:installButton.Enabled = $true
                $global:installButton.Text = "安装 WPS 2023"
                $global:installButton.BackColor = $primaryColor
            })
        }
        if ($global:install2019Button -ne $null) {
            $global:install2019Button.Invoke([Action]{
                $global:install2019Button.Enabled = $true
                $global:install2019Button.Text = "安装 WPS 2019"
                $global:install2019Button.BackColor = $secondaryColor
            })
        }
        [System.Windows.Forms.Application]::DoEvents()
        Hide-Progress
        return $false
    }
}

# =============================================
# 创建GUI界面
# =============================================

function Create-MainForm {
    # 创建主窗体
    $form = New-Object System.Windows.Forms.Form
    $form.Text = "WPS Office 2023/2019 专业增强版安装工具"
    $form.Size = New-Object System.Drawing.Size(950, 717)
    $form.StartPosition = "CenterScreen"
    $form.BackColor = $lightColor
    $form.FormBorderStyle = "FixedSingle"
    $form.MaximizeBox = $false
    $form.MinimizeBox = $true
    
    # 设置窗体图标
    try {
        $form.Icon = [System.Drawing.Icon]::ExtractAssociatedIcon((Get-Command powershell).Path)
    } catch {}
    
    # 头部面板
    $headerPanel = New-Object System.Windows.Forms.Panel
    $headerPanel.Location = New-Object System.Drawing.Point(0, 0)
    $headerPanel.Size = New-Object System.Drawing.Size(950, 100)
    $headerPanel.BackColor = $primaryColor
    $form.Controls.Add($headerPanel)
    
    # 标题标签
    $titleLabel = New-Object System.Windows.Forms.Label
    $titleLabel.Location = New-Object System.Drawing.Point(0, 20)
    $titleLabel.Size = New-Object System.Drawing.Size(950, 35)
    $titleLabel.Text = "WPS Office 2023/2019"
    $titleLabel.Font = $fontTitle
    $titleLabel.ForeColor = [System.Drawing.Color]::White
    $titleLabel.TextAlign = [System.Drawing.ContentAlignment]::MiddleCenter
    $headerPanel.Controls.Add($titleLabel)
    
    # 版本标签
    $versionLabel = New-Object System.Windows.Forms.Label
    $versionLabel.Location = New-Object System.Drawing.Point(0, 60)
    $versionLabel.Size = New-Object System.Drawing.Size(950, 20)
    $versionLabel.Text = "专业增强版 | 集成序列号 | 集成VBA | 去除升级功能"
    $versionLabel.Font = $fontSubtitle
    $versionLabel.ForeColor = [System.Drawing.Color]::FromArgb(220, 230, 255)
    $versionLabel.TextAlign = [System.Drawing.ContentAlignment]::MiddleCenter
    $headerPanel.Controls.Add($versionLabel)
    
    # 主内容面板
    $mainPanel = New-Object System.Windows.Forms.Panel
    $mainPanel.Location = New-Object System.Drawing.Point(0, 100)
    $mainPanel.Size = New-Object System.Drawing.Size(950, 550)
    $mainPanel.BackColor = $lightColor
    $form.Controls.Add($mainPanel)
    
    # 左侧控制面板
    $controlCard = New-Object System.Windows.Forms.GroupBox
    $controlCard.Location = New-Object System.Drawing.Point(20, 20)
    $controlCard.Size = New-Object System.Drawing.Size(300, 500)
    $controlCard.Text = "安装控制"
    $controlCard.Font = New-Object System.Drawing.Font($fontFamily, 10, [System.Drawing.FontStyle]::Bold)
    $controlCard.ForeColor = $textPrimary
    $controlCard.BackColor = [System.Drawing.Color]::White
    $mainPanel.Controls.Add($controlCard)
    
    # 功能按钮容器
    $buttonContainer = New-Object System.Windows.Forms.Panel
    $buttonContainer.Location = New-Object System.Drawing.Point(25, 40)
    $buttonContainer.Size = New-Object System.Drawing.Size(250, 400)
    $controlCard.Controls.Add($buttonContainer)
    
    # 功能按钮样式函数
    function Create-FeatureButton {
        param(
            [string]$Text,
            [System.Drawing.Color]$Color,
            [System.Drawing.Color]$HoverColor,
            [int]$YPosition,
            [scriptblock]$OnClick
        )
        
        $button = New-Object System.Windows.Forms.Button
        $button.Location = New-Object System.Drawing.Point(0, $YPosition)
        $button.Size = New-Object System.Drawing.Size(250, 40)
        $button.Text = $Text
        $button.Font = $fontButtonRegular
        $button.BackColor = $Color
        $button.ForeColor = [System.Drawing.Color]::White
        $button.FlatStyle = "Flat"
        $button.FlatAppearance.BorderSize = 0
        $button.FlatAppearance.MouseOverBackColor = $HoverColor
        $button.FlatAppearance.MouseDownBackColor = $Color
        $button.Cursor = [System.Windows.Forms.Cursors]::Hand
        $button.Add_Click($OnClick)
        
        return $button
    }
    
    # WPS 2023安装按钮
    $installButton = Create-FeatureButton -Text "安装 WPS Office 2023" -Color $primaryColor -HoverColor $primaryHover -YPosition 0 -OnClick {
        Install-WPS -Version "2023"
    }
    $buttonContainer.Controls.Add($installButton)
    $global:installButton = $installButton
    
    # WPS 2019安装按钮
    $install2019Button = Create-FeatureButton -Text "安装 WPS Office 2019" -Color $secondaryColor -HoverColor $secondaryHover -YPosition 50 -OnClick {
        Install-WPS -Version "2019"
    }
    $buttonContainer.Controls.Add($install2019Button)
    $global:install2019Button = $install2019Button
    
    # 查看2023信息按钮
    $info2023Button = Create-FeatureButton -Text "查看2023版本信息" -Color $successColor -HoverColor $successHover -YPosition 100 -OnClick {
        Show-Info -Version "2023"
    }
    $buttonContainer.Controls.Add($info2023Button)
    
    # 查看2019信息按钮
    $info2019Button = Create-FeatureButton -Text "查看2019版本信息" -Color $infoColor -HoverColor $infoHover -YPosition 150 -OnClick {
        Show-Info -Version "2019"
    }
    $buttonContainer.Controls.Add($info2019Button)
    
    # 检查安装按钮
    $checkButton = Create-FeatureButton -Text "检查现有安装" -Color $warningColor -HoverColor $warningHover -YPosition 200 -OnClick {
        # 直接调用详细模式，函数内部会输出完整信息
        $installInfo = Check-WPSInstalled -Detailed
    }
    $checkButton.ForeColor = [System.Drawing.Color]::Black
    $buttonContainer.Controls.Add($checkButton)
    
    # 清空日志按钮
    $clearButton = Create-FeatureButton -Text "清空日志" -Color $infoColor -HoverColor $infoHover -YPosition 250 -OnClick {
        if ($global:logTextBox -ne $null) {
            $global:logTextBox.Invoke([Action]{
                $global:logTextBox.Clear()
            })
            Write-Log "日志已清空" -Type "Info"
        }
    }
    $buttonContainer.Controls.Add($clearButton)
    
    # 退出按钮
    $exitButton = Create-FeatureButton -Text "退出程序" -Color $dangerColor -HoverColor $dangerHover -YPosition 300 -OnClick {
        $form.Close()
    }
    $buttonContainer.Controls.Add($exitButton)
    
    # 右侧日志面板
    $logCard = New-Object System.Windows.Forms.GroupBox
    $logCard.Location = New-Object System.Drawing.Point(340, 20)
    $logCard.Size = New-Object System.Drawing.Size(590, 500)
    $logCard.Text = "安装日志"
    $logCard.Font = New-Object System.Drawing.Font($fontFamily, 10, [System.Drawing.FontStyle]::Bold)
    $logCard.ForeColor = $textPrimary
    $logCard.BackColor = [System.Drawing.Color]::White
    $mainPanel.Controls.Add($logCard)
    
    # 进度条
    $progressBar = New-Object System.Windows.Forms.ProgressBar
    $progressBar.Location = New-Object System.Drawing.Point(15, 30)
    $progressBar.Size = New-Object System.Drawing.Size(560, 0)
    $progressBar.Visible = $false
    $progressBar.Minimum = 0
    $progressBar.Maximum = 100
    $progressBar.Style = 'Continuous'
    $logCard.Controls.Add($progressBar)
    $global:progressBar = $progressBar
    
    # 日志文本框
    $richTextBox = New-Object System.Windows.Forms.RichTextBox
    $richTextBox.Location = New-Object System.Drawing.Point(15, 35)
    $richTextBox.Size = New-Object System.Drawing.Size(560, 445)
    $richTextBox.BackColor = [System.Drawing.Color]::White
    $richTextBox.ReadOnly = $true
    $richTextBox.Font = $fontLog
    $richTextBox.BorderStyle = 'FixedSingle'
    $richTextBox.ScrollBars = 'Vertical'
    $logCard.Controls.Add($richTextBox)
    $global:logTextBox = $richTextBox
    
    # 底部状态栏
    $statusPanel = New-Object System.Windows.Forms.Panel
    $statusPanel.Location = New-Object System.Drawing.Point(0, 650)
    $statusPanel.Size = New-Object System.Drawing.Size(950, 30)
    $statusPanel.BackColor = $darkColor
    $form.Controls.Add($statusPanel)
    
    # 状态标签
    $statusLabel = New-Object System.Windows.Forms.Label
    $statusLabel.Location = New-Object System.Drawing.Point(10, 0)
    $statusLabel.Size = New-Object System.Drawing.Size(930, 30)
    $statusLabel.Text = "就绪 | 本软件仅供一切非商业性质的个人研究学习使用，任何用于商业或盈利的行为均被严格禁止。"
    $statusLabel.Font = New-Object System.Drawing.Font($fontFamily, 8.5)
    $statusLabel.ForeColor = [System.Drawing.Color]::White
    $statusLabel.TextAlign = [System.Drawing.ContentAlignment]::MiddleLeft
    $statusPanel.Controls.Add($statusLabel)
    
    # 版权标签
    $copyrightLabel = New-Object System.Windows.Forms.Label
    $copyrightLabel.Location = New-Object System.Drawing.Point(340, 525)
    $copyrightLabel.Size = New-Object System.Drawing.Size(590, 20)
    $copyrightLabel.Text = "© 2023-2024 WPS Office 专业增强版 "
    $copyrightLabel.Font = New-Object System.Drawing.Font($fontFamily, 8)
    $copyrightLabel.ForeColor = $textSecondary
    $copyrightLabel.TextAlign = [System.Drawing.ContentAlignment]::MiddleRight
    $mainPanel.Controls.Add($copyrightLabel)
    
    return $form
}

# =============================================
# 主程序
# =============================================

try {
    # 创建并显示窗体
    $global:form = Create-MainForm
    
    # 初始化日志
    $initialLog = @"
=== WPS Office 2023/2019 安装工具已启动 ===
支持版本: 
  • 2023专业增强版 (V12.8.2.21555)
  • 2019专业增强版 (V11.8.2.12300)
系统信息: Microsoft Windows 11 专业版
系统架构: 64位操作系统
当前用户: $env:USERNAME
系统时间: $(Get-Date -Format 'yyyy-MM-dd HH:mm:ss')

操作提示:
  • 点击'安装 WPS 2023'按钮进行完整安装
  • 点击'安装 WPS 2019'按钮安装2019版本
  • 自动注册序列号
  • 自动删除软件更新
准备就绪，请开始操作...

"@
    
    # 直接设置文本框内容
    $global:logTextBox.Text = $initialLog
    
    # 确保末尾有换行符
    if (-not $global:logTextBox.Text.EndsWith("`r`n")) {
        $global:logTextBox.AppendText("`r`n")
    }
    
    $global:logTextBox.SelectionStart = $global:logTextBox.TextLength
    $global:logTextBox.ScrollToCaret()
    
    # 显示窗体
    [void]$global:form.ShowDialog()
}
catch {
    $errorMsg = $_.Exception.Message
    [System.Windows.Forms.MessageBox]::Show("程序启动失败: $errorMsg", "错误", "OK", "Error")
}